#!/bin/bash
set -euo pipefail

SERVER_URL="http://localhost:8080"
SUBMISSION_ROOT="${SUBMISSION_ROOT:-$PWD}"
SERVER_DIR="${SERVER_DIR:-$SUBMISSION_ROOT}"
SERVER_LOG="${SERVER_LOG:-$SUBMISSION_ROOT/server.log}"
OUTPUT_DIR="${OUTPUT_DIR:-$SUBMISSION_ROOT/outputs}"
SERVER_PID=""

mkdir -p "$OUTPUT_DIR"

resolve_server_dir() {
  if [[ -f "$SERVER_DIR/Cargo.toml" ]]; then
    return
  fi

  local cargo_path
  cargo_path=$(find "$SERVER_DIR" -name Cargo.toml -print -quit 2>/dev/null || true)
  if [[ -z "$cargo_path" ]]; then
    echo "Error: Could not locate Cargo.toml under $SERVER_DIR" >&2
    exit 1
  fi
  SERVER_DIR=$(dirname "$cargo_path")
}

kill_port_processes() {
  if command -v lsof >/dev/null 2>&1; then
    local pids
    pids=$(lsof -ti tcp:8080 || true)
    for pid in $pids; do
      if [[ -n "$pid" ]]; then
        kill -9 "$pid" 2>/dev/null || true
      fi
    done
  fi
}

cleanup_state_files() {
  for base in "$SUBMISSION_ROOT" "$SERVER_DIR"; do
    if [[ -d "$base" ]]; then
      find "$base" \
        \( -name "*.db*" \
           -o -name "*.sqlite*" \
           -o -name "*.sqlite3*" \
           -o -name "*.*wal*" \
           -o -name "*.*shm*" \
           -o -name "*.txt" \
           -o -name "*.json" \
           -o -name "*library.toml" \
           -o -type d -name "*db*" \
           -o -type d -name "*sqlite*" \
           -o -type d -name "*database*" \
        \) -exec rm -rf {} + 2>/dev/null || true
    fi
  done
}

reset_state() {
  resolve_server_dir
  cleanup_state_files
  # Clearing server log...
  : >"$SERVER_LOG"
}

wait_for_server() {
  # Waiting for server to accept connections...
  for _ in $(seq 1 60); do
    if curl -s "$SERVER_URL" >/dev/null; then
      # Server is ready.
      return
    fi
    sleep 2
  done
  # Server failed to start within the timeout.
  exit 1
}

stop_server() {
  if [[ -n "${SERVER_PID}" ]] && kill -0 "$SERVER_PID" 2>/dev/null; then
    # Stopping server (PID $SERVER_PID)...
    kill "$SERVER_PID"
    wait "$SERVER_PID" 2>/dev/null || true
  fi
  SERVER_PID=""
}

restart_server() {
  stop_server
  kill_port_processes
  # Starting server from $SERVER_DIR ...
  pushd "$SERVER_DIR" >/dev/null
  cargo run --release >>"$SERVER_LOG" 2>&1 &
  SERVER_PID=$!
  popd >/dev/null
  wait_for_server
}

cleanup() {
  stop_server
  kill_port_processes
}

trap cleanup EXIT

# Start server fresh before running tests
reset_state
restart_server

#===============================================
# Test case 1: Concurrent Visit Handling
# -----------------------------------------------
echo "Testing case 1..."
{
  curl -s "$SERVER_URL"; echo;
  curl -s "$SERVER_URL/count"; echo;
  curl -s "$SERVER_URL/count"; echo;
  oha -n 1000000 "$SERVER_URL/count" >/dev/null
  curl -s "$SERVER_URL/count"; echo;
} >"$OUTPUT_DIR/case_1.txt" 2>>"$SERVER_LOG"

#===============================================
# Test case 2: Adding New Songs
# -----------------------------------------------
echo "Testing case 2..."
{
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Nebula", "artist": "Rihanna", "genre": "Electronic"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Horizon", "artist": "Ed Sheeran", "genre": "Blues"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Unity Resonance Cliff Galaxy Unity", "artist": "The Weeknd", "genre": "Country"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Radiance", "artist": "Adele", "genre": "Reggae"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Comet Quartz Spark Radiance", "artist": "Adele", "genre": "Country"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Flicker Breeze", "artist": "Rihanna", "genre": "Reggae"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Nebula Horizon Evergreen Resonance", "artist": "Sam Smith", "genre": "Jazz"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Wonder Unity Timeless Bliss Passage", "artist": "Post Malone", "genre": "Blues"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Wonder Blaze Passage Summit Blaze", "artist": "Katy Perry", "genre": "Folk"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Illusion", "artist": "Beyoncé", "genre": "Pop"}'; echo;
} >"$OUTPUT_DIR/case_2.txt" 2>>"$SERVER_LOG"

#===============================================
# Test case 3: Searching for Songs
# -----------------------------------------------
echo "Testing case 3..."
{
  curl -s "$SERVER_URL/songs/search?genre=Classical"; echo;
  curl -s "$SERVER_URL/songs/search?artist=Katy+Perry"; echo;
  curl -s "$SERVER_URL/songs/search?artist=The+Weeknd&genre=Folk"; echo;
  # curl -s "$SERVER_URL/songs/search?"; echo;
  curl -s "$SERVER_URL/songs/search?artist=Adele&genre=Folk"; echo;
  # curl -s "$SERVER_URL/songs/search?"; echo;
  curl -s "$SERVER_URL/songs/search?artist=Ariana+Grande&genre=Blues"; echo;
  curl -s "$SERVER_URL/songs/search?genre=Rock"; echo;
  curl -s "$SERVER_URL/songs/search?title=Fable+Serene+Frost&artist=Ariana+Grande"; echo;
  curl -s "$SERVER_URL/songs/search?artist=Dua+Lipa&genre=Electronic"; echo;
} >"$OUTPUT_DIR/case_3.txt" 2>>"$SERVER_LOG"

#===============================================
# Test case 4: Playing Songs
# -----------------------------------------------
echo "Testing case 4..."
{
  for song_id in $(seq 1 12); do
    curl -s "$SERVER_URL/songs/play/${song_id}"
    echo
  done
} >"$OUTPUT_DIR/case_4.txt" 2>>"$SERVER_LOG"

#===============================================
# Test case 5: Database Persistence
#-----------------------------------------------
echo "Testing case 5..."
# Restarting server before persistence checks...
restart_server
{
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Realm Thorn Treasure Resonance Eclipse", "artist": "Katy Perry", "genre": "Jazz"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Resonance Oasis Temple Journey Realm", "artist": "Harry Styles", "genre": "Rock"}'; echo;  
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Dew Iris Iris Wings Passage", "artist": "Justin Bieber", "genre": "Electronic"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Meadow Wave Dew Treasure Galaxy", "artist": "Shawn Mendes", "genre": "Country"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Symphony Shade Breeze Delight", "artist": "Selena Gomez", "genre": "Classical"}'; echo;

  # Restarting server...
  restart_server
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Shade Lullaby Temple", "artist": "Billie Eilish", "genre": "Reggae"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Passage", "artist": "Lil Nas X", "genre": "Reggae"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Resonance Velocity Crystal Rising", "artist": "Dua Lipa", "genre": "Hip-Hop"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Quartz Lagoon Spark Glimpse", "artist": "Justin Bieber", "genre": "Country"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Whispering Comet Shimmer Temple", "artist": "Beyoncé", "genre": "Folk"}'; echo;

  # Restarting server...
  restart_server
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Rising Ripple Frost Nebula", "artist": "Lil Nas X", "genre": "Hip-Hop"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Solitude", "artist": "Ed Sheeran", "genre": "Reggae"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Petal Stillness Evergreen Dawnlight", "artist": "The Weeknd", "genre": "Classical"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Illusion Unity Resonance", "artist": "Dua Lipa", "genre": "Folk"}'; echo;
  curl -s "$SERVER_URL/songs/new" --json '{"title": "Radiance Oasis Spectrum Infinity Bliss", "artist": "Sam Smith", "genre": "Country"}'; echo;

  # Final restart before validation queries...
  restart_server
  curl -s "$SERVER_URL/songs/search?title=Tide+Stream+Tranquil+Realm"; echo;
  curl -s "$SERVER_URL/songs/search?title=Radiance+Bliss+Dusk+Lullaby+Opal"; echo;
  curl -s "$SERVER_URL/songs/search?title=Cinder+Bliss+Cove+Iris&artist=Lady+Gaga"; echo;
  curl -s "$SERVER_URL/songs/search?artist=Dua+Lipa&genre=Reggae"; echo;
  curl -s "$SERVER_URL/songs/search?artist=Rihanna"; echo;
  curl -s "$SERVER_URL/songs/search?title=Iris+Lagoon&artist=The+Weeknd&genre=Rock"; echo;
  curl -s "$SERVER_URL/songs/search?genre=Jazz"; echo;
  curl -s "$SERVER_URL/songs/search?title=Ember+Cove+Glowstone&genre=Folk"; echo;
  curl -s "$SERVER_URL/songs/search?artist=Shawn+Mendes"; echo;
  curl -s "$SERVER_URL/songs/search?title=Orbit&artist=Sam+Smith"; echo;

  for song_id in $(seq 1 30); do
    curl -s "$SERVER_URL/songs/play/${song_id}"
    echo
  done
} >"$OUTPUT_DIR/case_5.txt" 2>>"$SERVER_LOG"